<?php
/**
 * Provides interface to store and retrieve plugin and user options
 * @subpackage Ad_Rpm_Booster_Options
 * @package Ad_Rpm_Booster
 */
class Ad_Rpm_Booster_Options {

	//default scope for options when called directly,
	//choices: site, user, or global (user option across sites)
	public $defaults = array();
	private $parent;

	/**
	 * Stores parent class as static
	 * @param class $parent (reference) the parent class
	 */
	function __construct( &$parent ) {

		$this->parent = &$parent;

		add_action( 'admin_init', array( &$this, 'options_init' ) );
		add_filter( $this->parent->prefix . 'options', array( &$this, 'default_options_filter' ), 20 );
		add_filter( $this->parent->prefix . 'js_options', array( &$this, 'db_version_filter' ) );

	}


	/**
	 * Tells WP that we're using a custom settings field
	 */
	function options_init() {

		register_setting( $this->parent->slug_, $this->parent->slug_, array( &$this, 'validate' ) );

	}


	/**
	 * Runs options through filter prior to saving
	 * @param array $options the options array
	 * @return array sanitized options array
	 */
	function validate( $options ) {



		//add slashes to JS selectors
		$js = array ( 'contentSelector', 'itemSelector', 'itemBodySelector', 'nextSelector' , 'navSelector' , 'displayOnlyLastPageSelector' );
		foreach ( $js as $field ) {

			if ( !isset( $options[$field] ) )
				continue;

			$options[$field] = addslashes( $options[ $field ] );

		}


		$options["flgAdInsertContentsLast"] = (bool)$options["flgAdInsertContentsLast"];
		$options["b_activate"] = (bool)$options["b_activate"];
		$options["b_output_rel_prev_next"] = (bool)$options["b_output_rel_prev_next"];
		$options["flg_ctrl_sidebar"] = (bool)$options["flg_ctrl_sidebar"];
		$options["flg_infinite_scroll"] = (bool)$options["flg_infinite_scroll"];


		$options["new_days"] = intval($options["new_days"]);
		if ($options["new_days"] < 1)	$options["new_days"] = 1;

		$options["judge_threshold_value"] = intval($options["judge_threshold_value"]);
		if ($options["judge_threshold_value"] < 1)	$options["judge_threshold_value"] = 1;

		$options["judge_short_threshold_value"] = intval($options["judge_short_threshold_value"]);
		if ($options["judge_short_threshold_value"] < 1)	$options["judge_short_threshold_value"] = 1;

		$options["split_first_length"] = intval($options["split_first_length"]);
		if ($options["split_first_length"] < 1)	$options["split_first_length"] = 1;

		$options["split_second_length"] = intval($options["split_second_length"]);
		if ($options["split_second_length"] < 1)	$options["split_second_length"] = 1;

		$options["max_split_num"] = intval($options["max_split_num"]);


		return apply_filters( $this->parent->prefix . 'options_validate', $options );

	}


	/**
	 * Allows overloading to get option value
	 * Usage: $value = $object->{option name}
	 * @param string $name the option name
	 * @return mixed the option value
	 */
	function __get( $name ) {

		return $this->get_option( $name );

	}


	/**
	 * Allows overloading to set option value
	 * Usage: $object->{option name} = $value
	 * @param string $name unique option key
	 * @param mixed $value the value to store
	 * @return bool success/fail
	 */
	function __set( $name, $value ) {

		return $this->set_option( $name, $value );

	}


	/**
	 * Retreive the options array
	 * @return array the options
	 */
	function get_options( ) {

		if ( !$options = wp_cache_get( 'options', $this->parent->slug ) ) {
			$options = get_option( $this->parent->slug_ );
			wp_cache_set( 'options', $options, $this->parent->slug );
		}

		return apply_filters( $this->parent->prefix . 'options', $options );
	}


	/**
	 * If any options are not set, merge with defaults
	 * @param array $options the saved options
	 * @return array the merged options with defaults
	 */
	function default_options_filter( $options ) {

		$options = wp_parse_args( $options,  $this->defaults );
		wp_cache_set( 'options', $options, $this->parent->slug );
		return $options;

	}


	/**
	 * Retreives a specific option
	 * @param string $option the unique option key
	 * @return mixed the value
	 */
	function get_option( $option ) {
		$options = $this->get_options( );
		$value = ( isset( $options[ $option ] ) ) ? $options[ $option ] : false;
		return apply_filters( $this->parent->prefix . $option, $value );
	}


	/**
	 * Sets a specific option
	 * @return bool success/fail
	 * @param string $key the unique option key
	 * @param mixed $value the value
	 */
	function set_option( $key, $value ) {
		$options = array( $key => $value );
		$this->set_options( $options );
	}


	/**
	 * Sets all plugin options
	 * @param array $options the options array
	 * @param bool $merge (optional) whether or not to merge options arrays or overwrite
	 * @return bool success/fail
	 */
	function set_options( $options, $merge = true ) {

		if ( $merge ) {
			$defaults = $this->get_options();
			$options = wp_parse_args( $options, $defaults );
		}

		wp_cache_set( 'options', $options, $this->parent->slug );

		return update_option( $this->parent->slug_, $options );

	}

	/**
	 * Don't output db_version to front end when passing args to javascript function
	 */
	function db_version_filter( $options ) {
		unset( $options['db_version'] );
		unset( $options['setup_time'] );
		unset( $options['activate_time'] );
		unset( $options['activate_hash'] );
		return $options;
	}

}
